package com.ejie.aa17a.service.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.i18n.LocaleContextHolder;

import com.ejie.aa17a.dao.iface.Aa17aAvisoDao;
import com.ejie.aa17a.dao.iface.Aa17aBandaDao;
import com.ejie.aa17a.dao.iface.Aa17aDatoHistoricoDao;
import com.ejie.aa17a.dao.iface.Aa17aDatoHorarioDao;
import com.ejie.aa17a.dao.iface.Aa17aEstacionDao;
import com.ejie.aa17a.model.Aa17aAviso;
import com.ejie.aa17a.model.Aa17aBanda;
import com.ejie.aa17a.model.Aa17aDatoHistorico;
import com.ejie.aa17a.model.Aa17aDatoHorario;
import com.ejie.aa17a.model.Aa17aEstacion;
import com.ejie.aa17a.model.geojson.Aa17aFeatures;
import com.ejie.aa17a.model.geojson.Aa17aGeojson;
import com.ejie.aa17a.model.geojson.Aa17aGeometry;
import com.ejie.aa17a.model.geojson.Aa17aProperties;
import com.ejie.aa17a.service.iface.Aa17aEstacionService;
import com.ejie.aa17a.utilidades.Aa17aConstantes;
import com.ejie.aa17a.utilidades.Aa17aFuncionesComunes;

/**
 * Aa17aEstacionServiceImpl generated by UDA, 18-dic-2013 15:36:14.
 * 
 * @author UDA
 */

public class Aa17aEstacionServiceImpl implements Aa17aEstacionService {

	private static final Logger logger = LoggerFactory.getLogger(Aa17aEstacionServiceImpl.class);

	private Aa17aEstacionDao aa17aEstacionDao;
	private Aa17aAvisoDao aa17aAvisoDao;
	private Aa17aDatoHorarioDao aa17aDatoHorarioDao;
	private Aa17aDatoHistoricoDao aa17aDatoHistoricoDao;
	private Aa17aBandaDao aa17aBandaDao;

	/**
	 * Finds a single row in the Aa17aEstacion table.
	 * 
	 * @param aa17aEstacion
	 *            Aa17aEstacion
	 * @return Aa17aEstacion
	 */
	public Aa17aEstacion find(Aa17aEstacion aa17aEstacion) {
		return (Aa17aEstacion) this.aa17aEstacionDao.find(aa17aEstacion);
	}

	/**
	 * Finds a List of rows in the Aa17aEstacion table.
	 * 
	 * @param aa17aEstacion
	 *            Aa17aEstacion
	 * @return List
	 */
	public List<Aa17aEstacion> findAll(Aa17aEstacion aa17aEstacion) {
		return (List<Aa17aEstacion>) this.aa17aEstacionDao.findAll(aa17aEstacion);
	}

	/**
	 * Devuelve un bean geojson con la lista de estaciones y sus coordenadas
	 * para la carga del mapa de la portada web
	 * 
	 * @param filterAa17aEstacion
	 *            Aa17aEstacion
	 * @return Aa17aGeojson
	 */
	public Aa17aGeojson getGeojsonMapa(Aa17aEstacion filterAa17aEstacion) {

		List<Aa17aEstacion> listEstaciones = this.aa17aEstacionDao.findAll(filterAa17aEstacion);

		Aa17aAviso avisoFiltro = new Aa17aAviso();
		avisoFiltro.setEstado(Aa17aConstantes.ESTADO_ACTIVO);

		Aa17aGeojson geojson = new Aa17aGeojson();
		geojson.setType(Aa17aConstantes.GEOJSON_TYPE_FEATURE_COLLECTION);

		Date fechaUltimaMedicion = null;
		Aa17aDatoHorario datoHorarioFiltro = new Aa17aDatoHorario();

		List<Aa17aFeatures> listFeatures = new ArrayList<Aa17aFeatures>();

		Aa17aBanda filtroBanda = new Aa17aBanda();
		Aa17aDatoHistorico datoHistoricoFiltro = new Aa17aDatoHistorico();

		// CHECKSTYLE:OFF CyclomaticComplexity - Generación de código de UDA
		for (Aa17aEstacion estacion : listEstaciones) {

			Aa17aFeatures features = new Aa17aFeatures();
			features.setType(Aa17aConstantes.GEOJSON_TYPE_FEATURE);

			Aa17aGeometry geometry = new Aa17aGeometry();
			geometry.setType(Aa17aConstantes.GEOJSON_TYPE_GEOMETRY);

			//coordenadas
			List<BigDecimal> coordenadas = new ArrayList<BigDecimal>();
			coordenadas.add(estacion.getCoordenadaX());
			coordenadas.add(estacion.getCoordenadaY());
			geometry.setCoordinates(coordenadas);

			//properties
			Aa17aProperties properties = new Aa17aProperties();
			Long idBanda = this.aa17aDatoHistoricoDao.getMaxIdBandaEstacion(estacion.getIdEstacion());
			if (idBanda == null) {
				idBanda = Aa17aConstantes.ID_BANDA_SIN_DATOS;
			}
			properties.setIdBanda(idBanda);
			properties.setTitle(estacion.getNombreEstacion());

			//datos de la banda
			filtroBanda.setIdBanda(idBanda);
			properties.setBanda(this.aa17aBandaDao.find(filtroBanda));

			//datos de la estacion
			properties.setEstacion(estacion);

			//obtengo los avisos
			avisoFiltro.setIdEstacion(estacion.getIdEstacion());
			properties.setAvisos(this.aa17aAvisoDao.findAll(avisoFiltro));

			//obtengo la fecha de la ultima medicion que existe para la estacion
			datoHorarioFiltro.setIdEstacion(estacion.getIdEstacion());
			fechaUltimaMedicion = this.aa17aDatoHorarioDao.getUltimaFechaMedida(datoHorarioFiltro);

			//si tengo fecha de ultima medicion
			if (fechaUltimaMedicion != null) {
				//obtengo los datos horarios de esta fecha para la estacion
				datoHorarioFiltro.setFechaDato(fechaUltimaMedicion);
				List<Aa17aDatoHorario> listDatosHorarios = this.aa17aDatoHorarioDao.findAll(datoHorarioFiltro);

				//establezco el filtro para obtener de la tabla de datos diarios el idBanda
				datoHistoricoFiltro.setIdEstacion(estacion.getIdEstacion());

				for (Aa17aDatoHorario datoHorario : listDatosHorarios) {

					//obtengo en formato String la fecha y hora de la última medición
					datoHorario.setFechaHoraUltimaMedidaHoraria(Aa17aFuncionesComunes.formatearFechaHora24(datoHorario.getUltimaMedidaHoraria(), LocaleContextHolder.getLocale()));

					/*
					 * si se trata de un parametro de tipo Contaminante y además tiene configurado un tipo de cálculo de ICA,
					 * obtengo los datos de la banda correspondiente
					 */
					if (datoHorario.getContaminante().getTipoParametro().equals(Aa17aConstantes.TIPO_PARAMETRO_CONTAMINANTE) &&
							datoHorario.getContaminante().getTipoCalculoICA() != null && datoHorario.getContaminante().getTipoCalculoICA() != Aa17aConstantes.MAGIC_NUMBER_0) {
						//establezco el filtro para obtener de la tabla de datos diarios el idBanda
						datoHistoricoFiltro.setContaminante(datoHorario.getContaminante());

						//obtengo el idBanda correspondiente al dia anterior para ese contaminante y estacion 
						Long idBandaHistorico = this.aa17aDatoHistoricoDao.getIdBandaContaminante(datoHistoricoFiltro);
						if (idBandaHistorico == null) {
							idBandaHistorico = Aa17aConstantes.ID_BANDA_SIN_DATOS;
						}

						//una vez tengo el idBanda, obtengo sus datos
						filtroBanda.setIdBanda(idBandaHistorico);
						datoHorario.setBanda(this.aa17aBandaDao.find(filtroBanda));
					}

				}

				properties.setDatosHorarios(listDatosHorarios);
			}

			features.setGeometry(geometry);
			features.setProperties(properties);

			listFeatures.add(features);

		}
		// CHECKSTYLE:ON CyclomaticComplexity - Generación de código de UDA
		geojson.setFeatures(listFeatures);

		return geojson;
	}

	/**
	 * Devuelve la lista de estaciones con la información de la banda
	 * correspondiente
	 * 
	 * @param aa17aEstacion
	 *            Aa17aEstacion
	 * @return List<Aa17aEstacion>
	 */
	public List<Aa17aEstacion> getListEstacionesConBanda(Aa17aEstacion aa17aEstacion) {

		List<Aa17aEstacion> listEstaciones = this.aa17aEstacionDao.findAll(aa17aEstacion);

		Aa17aBanda filtroBanda = new Aa17aBanda();

		for (Aa17aEstacion estacion : listEstaciones) {

			Long idBanda = this.aa17aDatoHistoricoDao.getMaxIdBandaEstacion(estacion.getIdEstacion());

			//si no tiene datos y no se puede obtener la banda
			if (idBanda == null) {
				idBanda = Aa17aConstantes.ID_BANDA_SIN_DATOS; //le asigno el idbanda de la banda sin datos
			}

			filtroBanda.setIdBanda(idBanda);
			estacion.setBanda(this.aa17aBandaDao.find(filtroBanda));
		}

		return listEstaciones;
	}

	/**
	 * Devuelve los datos de una estacion con la información de la banda
	 * correspondiente
	 * 
	 * @param aa17aEstacion
	 *            Aa17aEstacion
	 * @return Aa17aEstacion
	 */
	public Aa17aEstacion getEstacionConBanda(Aa17aEstacion aa17aEstacion) {

		Aa17aEstacion estacion = this.aa17aEstacionDao.find(aa17aEstacion);

		Long idBanda = this.aa17aDatoHistoricoDao.getMaxIdBandaEstacion(estacion.getIdEstacion());

		//si no tiene datos y no se puede obtener la banda
		if (idBanda == null) {
			idBanda = Aa17aConstantes.ID_BANDA_SIN_DATOS; //le asigno el idbanda de la banda sin datos
		}

		Aa17aBanda filtroBanda = new Aa17aBanda();
		filtroBanda.setIdBanda(idBanda);
		estacion.setBanda(this.aa17aBandaDao.find(filtroBanda));

		return estacion;
	}

	/**
	 * Setter method for Aa17aEstacionDao.
	 * 
	 * @param aa17aEstacionDao
	 *            Aa17aEstacionDao
	 * @return
	 */
	public void setAa17aEstacionDao(Aa17aEstacionDao aa17aEstacionDao) {
		Aa17aEstacionServiceImpl.logger.info("Setting Dependency " + aa17aEstacionDao);
		this.aa17aEstacionDao = aa17aEstacionDao;
	}

	/**
	 * @param aa17aAvisoDao
	 *            the aa17aAvisoDao to set
	 */
	public void setAa17aAvisoDao(Aa17aAvisoDao aa17aAvisoDao) {
		this.aa17aAvisoDao = aa17aAvisoDao;
	}

	/**
	 * @param aa17aDatoHorarioDao
	 *            the aa17aDatoHorarioDao to set
	 */
	public void setAa17aDatoHorarioDao(Aa17aDatoHorarioDao aa17aDatoHorarioDao) {
		this.aa17aDatoHorarioDao = aa17aDatoHorarioDao;
	}

	/**
	 * @param aa17aDatoHistoricoDao
	 *            the aa17aDatoHistoricoDao to set
	 */
	public void setAa17aDatoHistoricoDao(Aa17aDatoHistoricoDao aa17aDatoHistoricoDao) {
		this.aa17aDatoHistoricoDao = aa17aDatoHistoricoDao;
	}

	/**
	 * @param aa17aBandaDao
	 *            the aa17aBandaDao to set
	 */
	public void setAa17aBandaDao(Aa17aBandaDao aa17aBandaDao) {
		this.aa17aBandaDao = aa17aBandaDao;
	}

}
